<#
Parameter semantics: 
  - no parameter (or a value of 0) indicates default operation, 
  - a value of 1 indicates enhanced tracing

Script semantics:
  - saves parameter to CuttingBoard.parm,
  - reads preferences.xml (if present) to extract ArtifactDirectory, 
  - copies update\cuttingboard into current dir if present,
  - removes the update folder and several files under the ArtifactDirectory, and 
  - finally launches CuttingBoard.bat.
#>

param(
    [Parameter(Mandatory=$false)]
    [int]$logToFile = 0
)

# Write parameter file (creates or overwrites)
$parmFileName = "CuttingBoard.parm"
Set-Content -Path $parmFileName -Value $logToFile -Encoding ASCII

$prefFileName = "preferences.xml"
if (Test-Path $prefFileName) {
    # Read all lines
    $lines = Get-Content -Path $prefFileName -Raw -ErrorAction Stop

    # Regex to capture key and value similar to VBScript pattern:
    # matches something like: ... key="KEY" ... value="VALUE" />
    # The original VB regex is specific; adapt to find ArtifactDirectory="...".
    $artifactDir = $null

    # Look for ArtifactDirectory="...". Try a couple of common XML attribute layouts.
    # First try: ArtifactDirectory="some\path"
    if ($lines -match 'ArtifactDirectory\s*=\s*"([^"]+)"') {
        $artifactDir = $Matches[1]
    } else {
        # Generic attribute capture used in VBScript: use a more general pattern to find key/value pairs
        $pattern = '[^=]*="([^"]+)"[^"]*"([^"]+)"\/?>'
        $matches = [regex]::Matches($lines, $pattern)
        foreach ($m in $matches) {
            if ($m.Groups.Count -ge 3) {
                $key = $m.Groups[1].Value
                $value = $m.Groups[2].Value
                if ($key -eq 'ArtifactDirectory') {
                    $artifactDir = $value
                    break
                }
            }
        }
    }

    if ($artifactDir) {
        # Normalize path (expand environment variables if present)
        $dataDir = [Environment]::ExpandEnvironmentVariables($artifactDir)

        # Prepare update paths
        $updateDir = Join-Path $dataDir 'update'
        $cbDir = Join-Path $updateDir 'cuttingboard'

        if (Test-Path $cbDir) {
            try {
                # Copy contents of cbDir into current directory (overwriting)
                Copy-Item -Path (Join-Path $cbDir '*') -Destination (Get-Location) -Recurse -Force

                # Remove the update directory
                Remove-Item -Path $updateDir -Recurse -Force -ErrorAction SilentlyContinue

                # Remove specific files under dataDir if they exist
                $toDelete = @(
                    Join-Path $dataDir 'help\help.html',
                    Join-Path $dataDir 'import\DietaryReferenceTemplate.xml',
                    Join-Path $dataDir 'import\MealTemplate.xml',
                    Join-Path $dataDir 'import\RecipeTemplate.xml'
                )
                foreach ($f in $toDelete) {
                    if (Test-Path $f) {
                        Remove-Item -Path $f -Force -ErrorAction SilentlyContinue
                    }
                }
            } catch {
                Write-Warning "Update step encountered an error: $($_.Exception.Message)"
            }
        }
    }
}

# Launch CuttingBoard.bat hidden (no visible window)
$bat = Join-Path (Get-Location) 'CuttingBoard.bat'
if (Test-Path $bat) {
    $si = New-Object System.Diagnostics.ProcessStartInfo
    $si.FileName = $bat
    $si.WorkingDirectory = (Get-Location).Path
    $si.WindowStyle = 'Hidden'
    $si.CreateNoWindow = $true
    $si.UseShellExecute = $true

    [System.Diagnostics.Process]::Start($si) | Out-Null
} else {
    Write-Warning "CuttingBoard.bat not found in current directory."
}